using System;
using Server.Gumps;
using Server.Multis;
using Server.Network;

namespace Server.Items
{
	public class ElvenDisplayCaseAddon : BaseAddon
	{
		public override BaseAddonDeed Deed
		{
			get
			{
				ElvenDisplayCaseDeed deed = new ElvenDisplayCaseDeed();
				deed.Dusts = m_Dusts;

				return deed;
			}
		}
		
		private int m_Dusts;
		private Timer m_Timer;

		[CommandProperty( AccessLevel.GameMaster )]
		public int Dusts
		{
			get{ return m_Dusts; }
			set{ m_Dusts = value; InvalidateProperties(); }
		}

		[Constructable]
		public ElvenDisplayCaseAddon( bool east ) : base()
		{
			if ( east ) // east
			{
				AddComponent( new AddonComponent( 12391 ), 0, 0, 0 );
				AddComponent( new AddonComponent( 12390 ), 0, +1, 0 );
				AddComponent( new AddonComponent( 12392 ), 0, -1, 0 );
			}
			else 		// south
			{
				AddComponent( new AddonComponent( 12394 ), 0, 0, 0 );
				AddComponent( new AddonComponent( 12393 ), +1, 0, 0 );
				AddComponent( new AddonComponent( 12395 ), -1, 0, 0 );
			}
			
			m_Timer = Timer.DelayCall( TimeSpan.FromDays( 1 ), TimeSpan.FromDays( 1 ), new TimerCallback( GiveResources ) );
		}
		
		private void GiveResources()
		{
			m_Dusts = Math.Min( 2000, m_Dusts + 250 );
		}
		
		public override void OnComponentUsed( AddonComponent c, Mobile from )
		{
			BaseHouse house = BaseHouse.FindHouseAt( this );

			if (!from.InRange(GetWorldLocation(), 2) || !from.InLOS(this) || !((from.Z - Z) > -3 && (from.Z - Z) < 3))
			{
				from.LocalOverheadMessage(Network.MessageType.Regular, 0x3B2, 1019045); // I can't reach that.
			}
			else if ( house != null && house.HasSecureAccess( from, SecureLevel.Friends ) )
			{
				if ( m_Dusts > 0 )
				{
    				Item dusts = null;

					switch ( Utility.Random( 4 ) )
					{
						case 0: dusts = new Xanthos.Evo.RaelisDemonioDust(); break;
						case 1: dusts = new Xanthos.Evo.RaelisDragonDust(); break;
						case 2: dusts = new Xanthos.Evo.HiryuEvoDust(); break;
						case 3: dusts = new Xanthos.Evo.EvoSpiderDust(); break;
					}

					int amount = Math.Min( 600, m_Dusts );
					dusts.Amount = amount;

					if ( !from.PlaceInBackpack( dusts ) )
					{
						dusts.Delete();
						from.SendLocalizedMessage( 1078837 ); // Your backpack is full! Please make room and try again.
					}
					else
					{
						m_Dusts -= amount;
						PublicOverheadMessage( MessageType.Regular, 0, false, String.Format( "Dusts: {0}", m_Dusts ) );
					}
				}
				else
					from.SendMessage( "There are no more resources available at this time." );
			}
			else
				from.SendLocalizedMessage( 1061637 ); // You are not allowed to access this.
		}

		public ElvenDisplayCaseAddon( Serial serial ) : base( serial )
		{
		}

		public override void Serialize( GenericWriter writer )
		{
			base.Serialize( writer );

			writer.WriteEncodedInt( 0 ); // version

			writer.Write( (int) m_Dusts );

			if ( m_Timer != null )
				writer.Write( (DateTime) m_Timer.Next );
			else
				writer.Write( (DateTime) DateTime.Now + TimeSpan.FromDays( 1 ) );
		}

		public override void Deserialize( GenericReader reader )
		{
			base.Deserialize( reader );

			int version = reader.ReadEncodedInt();

			m_Dusts = reader.ReadInt();

			DateTime next = reader.ReadDateTime();

			if ( next < DateTime.Now )
				next = DateTime.Now;

			m_Timer = Timer.DelayCall( next - DateTime.Now, TimeSpan.FromDays( 1 ), new TimerCallback( GiveResources ) );
		}
	}

	public class ElvenDisplayCaseDeed : BaseAddonDeed
	{
		public override BaseAddon Addon
		{
			get
			{
				ElvenDisplayCaseAddon addon = new ElvenDisplayCaseAddon( m_East );
				addon.Dusts = m_Dusts;

				return addon;
			}
		}

		private bool m_East;
		private int m_Dusts;

		[CommandProperty( AccessLevel.GameMaster )]
		public int Dusts
		{
			get{ return m_Dusts; }
			set{ m_Dusts = value; InvalidateProperties(); }
		}

		[Constructable]
		public ElvenDisplayCaseDeed() : base()
		{
			LootType = LootType.Blessed;
			Name = "elven display case";
		}

		public ElvenDisplayCaseDeed( Serial serial ) : base( serial )
		{
		}

		public override void OnDoubleClick( Mobile from )
		{
			if ( IsChildOf( from.Backpack ) )
			{
				from.CloseGump( typeof( InternalGump ) );
				from.SendGump( new InternalGump( this ) );
			}
			else
				from.SendLocalizedMessage( 1062334 ); // This item must be in your backpack to be used.
		}

		private void SendTarget( Mobile m )
		{
			base.OnDoubleClick( m );
		}

		public override void Serialize( GenericWriter writer )
		{
			base.Serialize( writer );

			writer.WriteEncodedInt( 0 ); // version

			writer.Write( (int) m_Dusts );
		}

		public override void Deserialize( GenericReader reader )
		{
			base.Deserialize( reader );

			int version = reader.ReadEncodedInt();

			m_Dusts = reader.ReadInt();
		}

		private class InternalGump : Gump
		{
			private ElvenDisplayCaseDeed m_Deed;

			public InternalGump( ElvenDisplayCaseDeed deed ) : base( 60, 36 )
			{
				m_Deed = deed;

				AddPage( 0 );

				AddBackground( 0, 0, 273, 324, 0x13BE );
				AddImageTiled( 10, 10, 253, 20, 0xA40 );
				AddImageTiled( 10, 40, 253, 244, 0xA40 );
				AddImageTiled( 10, 294, 253, 20, 0xA40 );
				AddAlphaRegion( 10, 10, 253, 304 );
				AddButton( 10, 294, 0xFB1, 0xFB2, 0, GumpButtonType.Reply, 0 );
				AddHtmlLocalized( 45, 296, 450, 20, 1060051, 0x7FFF, false, false ); // CANCEL
				AddHtmlLocalized( 14, 12, 273, 20, 1076579, 0x7FFF, false, false ); // Please select your statue position

				AddPage( 1 );

				AddButton( 19, 49, 0x845, 0x846, 1, GumpButtonType.Reply, 0 );
				AddHtmlLocalized( 44, 47, 213, 20, 1075386, 0x7FFF, false, false ); // South
				AddButton( 19, 73, 0x845, 0x846, 2, GumpButtonType.Reply, 0 );
				AddHtmlLocalized( 44, 71, 213, 20, 1075387, 0x7FFF, false, false ); // East
			}

			public override void OnResponse( NetState sender, RelayInfo info )
			{
				if ( m_Deed == null || m_Deed.Deleted || info.ButtonID == 0 )
					return;

				m_Deed.m_East = ( info.ButtonID != 1 );
				m_Deed.SendTarget( sender.Mobile );
			}
		}
	}
}